<?php
declare(strict_types=1);

require_once __DIR__ . '/../core.php';

class RedgifsFetcher implements SiteFetcher {
    public static function canHandle(string $url): bool {
        return preg_match('~redgifs\.com/watch/~i', $url) === 1;
    }

    public static function fetchMedia(string $url): array {
        if (!preg_match('~redgifs\.com/watch/([a-zA-Z0-9]+)~', $url, $m)) {
            throw new RuntimeException('❌ لینک Redgifs معتبر نیست: ' . $url);
        }
        $slug = $m[1];

        $tokenJson = httpGet('https://api.redgifs.com/v2/auth/temporary', [
            "User-Agent: redownloader/1.0 by reza",
            "Accept: application/json"
        ]);
        $tokenData = json_decode($tokenJson, true);
        if (empty($tokenData['token'])) throw new RuntimeException('❌ توکن در پاسخ وجود ندارد.');
        $token = $tokenData['token'];

        $videoJson = httpGet("https://api.redgifs.com/v2/gifs/$slug", [
            "Authorization: Bearer $token",
            "User-Agent: redownloader/1.0 by reza",
            "Accept: application/json"
        ]);
        $videoData = json_decode($videoJson, true);
        if (empty($videoData['gif'])) throw new RuntimeException('❌ پاسخ API فاقد کلید gif است.');

        $gif = $videoData['gif'];
        $src = $gif['urls']['hd']
            ?? $gif['urls']['sd']
            ?? $gif['urls']['download']
            ?? $gif['urls']['gif']
            ?? null;

        if (!$src) throw new RuntimeException('❌ هیچ لینک ویدئویی در پاسخ وجود ندارد.');

        $title  = sanitizeSlug($gif['title'] ?? $slug);
        $author = sanitizeSlug($gif['userName'] ?? 'redgifs');
        $ext    = pathinfo(parse_url($src, PHP_URL_PATH) ?? '', PATHINFO_EXTENSION) ?: 'mp4';

        return [[
            'url'    => $src,
            'type'   => 'video',
            'author' => $author,
            'title'  => $title,
            'ext'    => $ext
        ]];
    }
}