// scripts.js

document.addEventListener("DOMContentLoaded", () => {
    /* ===========================
       اسپیدومتر (Gauge)
       =========================== */
    function drawGauge(canvas, percent, label, color) {
        const ctx = canvas.getContext("2d");
        const radius = canvas.width / 2 - 10;
        ctx.clearRect(0, 0, canvas.width, canvas.height);

        // زمینه
        ctx.beginPath();
        ctx.arc(canvas.width/2, canvas.height/2, radius, Math.PI, 0);
        ctx.strokeStyle = "#444";
        ctx.lineWidth = 20;
        ctx.stroke();

        // مقدار
        ctx.beginPath();
        ctx.arc(canvas.width/2, canvas.height/2, radius, Math.PI, Math.PI + (percent/100)*Math.PI);
        ctx.strokeStyle = color;
        ctx.lineWidth = 20;
        ctx.stroke();

        // متن
        ctx.fillStyle = "#fff";
        ctx.font = "14px Vazirmatn, sans-serif";
        ctx.textAlign = "center";
        ctx.fillText(label, canvas.width/2, canvas.height/2 + 40);
        ctx.fillText(percent + "%", canvas.width/2, canvas.height/2 + 60);
    }

    /* ===========================
       مدیریت فرم دانلود با AJAX
       =========================== */
    const form = document.getElementById("downloadForm");
    const progressSection = document.getElementById("progressSection");
    const progressInfo = document.getElementById("progressInfo");
    const speedGauge = document.getElementById("speedGauge");
    const progressGauge = document.getElementById("progressGauge");

    if (form) {
        form.addEventListener("submit", e => {
            e.preventDefault();
            const formData = new FormData(form);

            progressSection.classList.remove("hidden");
            progressInfo.textContent = "در حال آماده‌سازی دانلود...";

            fetch("download.php", {
                method: "POST",
                body: formData
            })
                .then(r => r.json())
                .then(data => {
                    if (data.error) {
                        progressInfo.textContent = "❌ خطا: " + data.error;
                        return;
                    }
                    if (data.requireContentType) {
                        progressInfo.textContent = "لطفاً نوع محتوا را انتخاب کنید.";
                        return;
                    }

                    progressInfo.textContent = "دانلود آغاز شد...";
                    const progressId = data.progressId;
                    const redirectUrl = data.redirect;

                    const interval = setInterval(() => {
                        fetch("progress.php?id=" + encodeURIComponent(progressId))
                            .then(r => r.json())
                            .then(p => {
                                if (!p || !p.total) return;
                                const percent = Math.round((p.downloaded / p.total) * 100);
                                const speed = p.speed || 0;

                                drawGauge(speedGauge, Math.min(speed, 100), "سرعت", "#22c55e");
                                drawGauge(progressGauge, percent, "پیشرفت", "#3b82f6");

                                progressInfo.textContent = `دانلود شده: ${p.downloaded_str} / ${p.total_str}`;

                                if (percent >= 100) {
                                    clearInterval(interval);
                                    progressInfo.textContent = "✅ دانلود کامل شد!";
                                    setTimeout(() => {
                                        window.location.href = redirectUrl;
                                    }, 1500);
                                }
                            })
                            .catch(err => {
                                clearInterval(interval);
                                progressInfo.textContent = "❌ خطا در مانیتورینگ: " + err;
                            });
                    }, 2000);
                })
                .catch(err => {
                    progressInfo.textContent = "❌ خطا: " + err;
                });
        });
    }

    /* ===========================
       پنجره تأیید حذف
       =========================== */
    function openConfirm(text, onOk) {
        const modal = document.getElementById('confirmModal');
        const txt = document.getElementById('confirmText');
        const ok = document.getElementById('confirmOk');
        const cancel = document.getElementById('confirmCancel');

        if (!modal) return alert(text);

        txt.textContent = text;
        modal.classList.remove('hidden');
        modal.setAttribute('aria-hidden', 'false');

        const onOkWrap = () => {
            ok.removeEventListener('click', onOkWrap);
            cancel.removeEventListener('click', onCancelWrap);
            modal.classList.add('hidden');
            modal.setAttribute('aria-hidden', 'true');
            onOk && onOk();
        };
        const onCancelWrap = () => {
            ok.removeEventListener('click', onOkWrap);
            cancel.removeEventListener('click', onCancelWrap);
            modal.classList.add('hidden');
            modal.setAttribute('aria-hidden', 'true');
        };

        ok.addEventListener('click', onOkWrap);
        cancel.addEventListener('click', onCancelWrap);
    }

    function deleteRequest(action, payload, onDone) {
        const params = new URLSearchParams();
        params.set('action', action);

        if (action === 'file') {
            params.set('target', payload.target);
        } else if (action === 'link') {
            params.set('author', payload.author);
            params.set('linkId', payload.linkId);
        } else if (action === 'author') {
            params.set('author', payload.author);
        }

        fetch('delete.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: params.toString()
        }).then(r => r.json())
            .then(res => {
                if (res.error) {
                    alert('❌ خطا: ' + res.error);
                    return;
                }
                alert(res.message || (res.success ? '✅ حذف شد.' : '❌ حذف انجام نشد.'));
                onDone && onDone();
            }).catch(err => alert('❌ خطا در حذف: ' + err));
    }

    /* ===========================
       اتصال دکمه‌های حذف در گالری
       =========================== */
    document.querySelectorAll('.btn-delete-file').forEach(btn => {
        btn.addEventListener('click', e => {
            e.preventDefault();
            e.stopPropagation();
            const file = btn.dataset.file;
            if (!file) return;
            openConfirm('آیا مطمئن هستی که می‌خواهی این فایل را حذف کنی؟', () => {
                deleteRequest('file', { target: file }, () => location.reload());
            });
        });
    });

    document.querySelectorAll('.btn-delete-link').forEach(btn => {
        btn.addEventListener('click', e => {
            e.preventDefault();
            e.stopPropagation();
            const author = btn.dataset.author;
            const linkId = btn.dataset.linkid;
            if (!author || !linkId) return;
            openConfirm('آیا مطمئن هستی که می‌خواهی کل محتوای این لینک را حذف کنی؟', () => {
                deleteRequest('link', { author, linkId }, () => {
                    window.location.href = 'gallery.php?author=' + encodeURIComponent(author);
                });
            });
        });
    });

    document.querySelectorAll('.btn-delete-author').forEach(btn => {
        btn.addEventListener('click', e => {
            e.preventDefault();
            const author = btn.dataset.author;
            if (!author) return;
            openConfirm('آیا مطمئن هستی که می‌خواهی کل محتوای این سایت را حذف کنی؟', () => {
                deleteRequest('author', { author }, () => {
                    window.location.href = 'gallery.php';
                });
            });
        });
    });
});

