<?php
declare(strict_types=1);

require_once __DIR__ . '/../core.php';

class GenericFetcher implements SiteFetcher {
    public static function canHandle(string $url): bool {
        // همیشه false تا فقط به‌عنوان fallback در رجیستری استفاده شود
        return false;
    }

    // پارامتر دوم اختیاری برای نوع محتوا (image/video/audio/gif/pdf)
    public static function fetchMedia(string $url, ?string $contentType = null): array {
        // اگر لینک مستقیم فایل باشد، همان را برگردان
        $extPath = parse_url($url, PHP_URL_PATH) ?? '';
        $ext = strtolower(pathinfo($extPath, PATHINFO_EXTENSION));
        $type = self::detectType($ext);

        if ($type !== 'other') {
            return [[
                'url'    => $url,
                'type'   => $type,
                'author' => sanitizeSlug(parse_url($url, PHP_URL_HOST) ?? 'unknown'),
                'title'  => sanitizeSlug(basename($extPath ?: 'file')),
                'ext'    => $ext ?: self::defaultExtForType($type),
            ]];
        }

        if (!$contentType) {
            throw new RuntimeException('GenericFetcher نیاز به contentType دارد.');
        }

        // دانلود صفحه و استخراج لینک‌های مدیا
        $headers = [
            "User-Agent: Mozilla/5.0 (GenericFetcher/1.0)",
            "Referer: $url",
            "Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
        ];
        $html = httpGet($url, $headers);
        if (!$html) throw new RuntimeException("❌ محتوای صفحه دریافت نشد.");

        $author = sanitizeSlug(parse_url($url, PHP_URL_HOST) ?? 'unknown');
        $title  = sanitizeSlug(basename(parse_url($url, PHP_URL_PATH) ?? 'page'));
        $list   = [];

        // <a href> با <img> داخلی
        if (preg_match_all('~<a[^>]+href=["\']([^"\']+)["\'][^>]*>(.*?)</a>~is', $html, $am)) {
            foreach ($am[1] as $i => $href) {
                $inner = $am[2][$i] ?? '';
                if (preg_match('~<img[^>]+src=["\']([^"\']+)["\']~i', $inner)) {
                    $abs = absoluteUrl($url, html_entity_decode($href));
                    self::pushIfTypeMatches($list, $abs, $contentType, $author, $title);
                }
            }
        }

        // همه‌ی <img src>
        if (preg_match_all('~<img[^>]+src=["\']([^"\']+)["\']~i', $html, $im)) {
            foreach ($im[1] as $src) {
                $abs = absoluteUrl($url, html_entity_decode($src));
                self::pushIfTypeMatches($list, $abs, $contentType, $author, $title);
            }
        }

        // لینک‌های مستقیم فایل‌ها در <a href>
        if (preg_match_all('~<a[^>]+href=["\']([^"\']+)["\']~i', $html, $lm)) {
            foreach ($lm[1] as $href) {
                $abs = absoluteUrl($url, html_entity_decode($href));
                self::pushIfTypeMatches($list, $abs, $contentType, $author, $title);
            }
        }

        // حذف موارد تکراری
        $seen = [];
        $uniq = [];
        foreach ($list as $it) {
            if (isset($seen[$it['url']])) continue;
            $seen[$it['url']] = true;
            $uniq[] = $it;
        }

        if (empty($uniq)) throw new RuntimeException("❌ هیچ محتوای {$contentType} یافت نشد.");
        return $uniq;
    }

    private static function pushIfTypeMatches(array &$list, string $absUrl, string $wantType, string $author, string $title): void {
        $ext  = strtolower(pathinfo(parse_url($absUrl, PHP_URL_PATH) ?? '', PATHINFO_EXTENSION));
        $type = self::detectType($ext);

        if ($type === $wantType) {
            $list[] = [
                'url'    => $absUrl,
                'type'   => $type,
                'author' => $author,
                'title'  => $title,
                'ext'    => $ext ?: self::defaultExtForType($type),
            ];
        }
    }

    private static function detectType(string $ext): string {
        $ext = strtolower($ext);
        $images = ['jpg','jpeg','png','webp'];
        $gifs   = ['gif'];
        $videos = ['mp4','webm','m4v','mov'];
        $audios = ['mp3','wav','ogg','m4a'];
        $pdfs   = ['pdf'];

        if (in_array($ext, $images)) return 'image';
        if (in_array($ext, $gifs))   return 'gif';
        if (in_array($ext, $videos)) return 'video';
        if (in_array($ext, $audios)) return 'audio';
        if (in_array($ext, $pdfs))   return 'pdf';
        return 'other';
    }

    private static function defaultExtForType(string $type): string {
        switch ($type) {
            case 'image': return 'jpg';
            case 'gif':   return 'gif';
            case 'video': return 'mp4';
            case 'audio': return 'mp3';
            case 'pdf':   return 'pdf';
            default:      return 'bin';
        }
    }
}