<?php
declare(strict_types=1);

/* ===========================
   تنظیمات و مسیرها (Config)
   =========================== */

define('STORAGE_PATH', __DIR__ . '/storage');
define('MEDIA_PATH',   STORAGE_PATH . '/media');
define('THUMB_PATH',   STORAGE_PATH . '/thumbs');
define('META_PATH',    STORAGE_PATH . '/meta');
define('PROGRESS_PATH', STORAGE_PATH . '/progress'); // وضعیت لحظه‌ای دانلود برای AJAX

define('ASSETS_PATH', __DIR__ . '/assets');
define('LOGOS_PATH',  ASSETS_PATH . '/logos');       // ذخیره‌ی لوگو/فاوآیکون سایت‌ها

define('CA_BUNDLE', __DIR__ . '/assets/cacert.pem');
define('MAX_DOWNLOADS_PER_POST', 50);
define('TIMEOUT_SECONDS', 400);

// اطلاعات اپلیکیشن Reddit (برای پلاگین Reddit)
define('REDDIT_CLIENT_ID',     'LaX44OR4_eLCYGaKyZiSBg');
define('REDDIT_CLIENT_SECRET', 'ePBp2g-1b0ZPs-4zzFfovYNlVQLkhw');
define('REDDIT_USERNAME',      'fbbongafb');
define('REDDIT_PASSWORD',      'TAn1101101369629');

// ساخت پوشه‌های اصلی
foreach ([STORAGE_PATH, MEDIA_PATH, THUMB_PATH, META_PATH, PROGRESS_PATH, LOGOS_PATH] as $dir) {
    if (!is_dir($dir)) mkdir($dir, 0775, true);
}

/* ===========================
   عیبیابی (Logging)
   =========================== */
function logMessage(string $message): void {
    $logFile = __DIR__ . '/process.log';
    $date = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$date] $message\n", FILE_APPEND);
}

/* ===========================
   اینترفیس پلاگین‌ها
   =========================== */

interface SiteFetcher {
    public static function canHandle(string $url): bool;
    // هر آیتم: ['url','type','author','title','ext', optional: 'audio_url']
    public static function fetchMedia(string $url): array;
}

/* ===========================
   ابزارهای فضای هاست و فرمت‌ها
   =========================== */

function formatBytes(float $bytes): string {
    $units = ['B','KB','MB','GB','TB'];
    $i = 0;
    while ($bytes >= 1024 && $i < count($units)-1) {
        $bytes /= 1024;
        $i++;
    }
    return round($bytes, 2) . ' ' . $units[$i];
}

function getHostDiskUsage(): array {
    $root = __DIR__;
    $total = @disk_total_space($root) ?: 0;
    $free  = @disk_free_space($root) ?: 0;
    $used  = max(0, $total - $free);
    $percent = $total > 0 ? round(($used / $total) * 100, 2) : 0.0;
    return [
        'total'   => $total,
        'free'    => $free,
        'used'    => $used,
        'percent' => $percent,
        'total_str' => formatBytes($total),
        'free_str'  => formatBytes($free),
        'used_str'  => formatBytes($used),
    ];
}

/* ===========================
   توابع کمکی عمومی (Helpers)
   =========================== */

function sanitizeSlug(string $input): string {
    $slug = preg_replace('~[^\pL\d]+~u', '-', $input);
    $slug = @iconv('UTF-8', 'ASCII//TRANSLIT', $slug) ?: $input;
    $slug = preg_replace('~[^-\w]+~', '', $slug);
    $slug = trim($slug, '-');
    $slug = preg_replace('~-+~', '-', $slug);
    return strtolower($slug ?: 'post');
}

function absoluteUrl(string $base, string $path): string {
    if ($path === '') return $base;
    if (preg_match('~^https?://~i', $path)) return $path;
    if (strpos($path, '//') === 0) {
        $scheme = parse_url($base, PHP_URL_SCHEME) ?: 'https';
        return $scheme . ':' . $path;
    }
    $parts = parse_url($base);
    $scheme   = $parts['scheme'] ?? 'https';
    $host     = $parts['host'] ?? '';
    $port     = isset($parts['port']) ? ':' . $parts['port'] : '';
    $basePath = $parts['path'] ?? '/';
    if ($path[0] !== '/') {
        if (substr($basePath, -1) !== '/') {
            $basePath = preg_replace('~/[^/]+$~', '/', $basePath);
        }
        $path = $basePath . $path;
    }
    $segments = [];
    foreach (explode('/', $path) as $seg) {
        if ($seg === '' || $seg === '.') continue;
        if ($seg === '..') array_pop($segments);
        else $segments[] = $seg;
    }
    $normalized = '/' . implode('/', $segments);
    return $scheme . '://' . $host . $port . $normalized;
}

function httpGet(string $url, array $headers = []): string {
    $ch = curl_init($url);
    $ref = '';
    $pu = parse_url($url);
    if (!empty($pu['scheme']) && !empty($pu['host'])) {
        $ref = $pu['scheme'] . '://' . $pu['host'];
    }
    $defaultHeaders = ['User-Agent: Mozilla/5.0 (Downloader)', 'Accept: */*'];
    if ($ref) $defaultHeaders[] = 'Referer: ' . $ref;
    $opts = [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => TIMEOUT_SECONDS,
        CURLOPT_HTTPHEADER     => array_merge($defaultHeaders, $headers),
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT      => 'Mozilla/5.0 (Downloader)',
    ];
    if (file_exists(CA_BUNDLE)) {
        $opts[CURLOPT_SSL_VERIFYPEER] = true;
        $opts[CURLOPT_CAINFO]         = CA_BUNDLE;
    } else {
        $opts[CURLOPT_SSL_VERIFYPEER] = false;
    }
    curl_setopt_array($ch, $opts);
    $data = curl_exec($ch);
    $err  = curl_error($ch);
    curl_close($ch);
    if ($data === false) throw new RuntimeException("Failed to GET $url: $err");
    return $data;
}

/* ===========================
   مدیریت دانلود با گزارش پیشرفت (برای AJAX)
   =========================== */

/**
 * دانلود فایل با ثبت پیشرفت در فایل JSON (برای نمایش نوار پیشرفت در فرانت‌اند).
 * @param string $url       آدرس فایل
 * @param string $destPath  مسیر مقصد
 * @param string|null $progressId شناسه‌ی پیشرفت (مثلاً md5 لینک مادر). اگر null باشد، پیشرفت ثبت نمی‌شود.
 */
function downloadFile(string $url, string $destPath, ?string $progressId = null): void {
    $dir = dirname($destPath);
    if (!is_dir($dir)) mkdir($dir, 0775, true);
    $tmp = $destPath . '.part';
    $fp  = fopen($tmp, 'wb');
    if (!$fp) throw new RuntimeException("Cannot write temp file: $tmp");

    $ch = curl_init($url);
    $ref = '';
    $pu = parse_url($url);
    if (!empty($pu['scheme']) && !empty($pu['host'])) {
        $ref = $pu['scheme'] . '://' . $pu['host'];
    }
    $headers = ['Accept: */*'];
    if ($ref) $headers[] = 'Referer: ' . $ref;

    $opts = [
        CURLOPT_FILE           => $fp,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT        => TIMEOUT_SECONDS,
        CURLOPT_USERAGENT      => 'Mozilla/5.0 (Downloader)',
        CURLOPT_HTTPHEADER     => $headers,
    ];
    // ثبت پیشرفت با XFERINFO (PHP 7.1+) یا PROGRESSFUNCTION (قدیمی‌تر)
    if ($progressId) {
        $progressFile = PROGRESS_PATH . '/' . sanitizeSlug($progressId) . '.json';
        $state = ['downloaded' => 0, 'total' => 0, 'updated_at' => time()];
        file_put_contents($progressFile, json_encode($state));

        // فعال کردن گزارش پیشرفت
        $opts[CURLOPT_NOPROGRESS] = false;
        if (defined('CURLOPT_XFERINFOFUNCTION')) {
            $opts[CURLOPT_XFERINFOFUNCTION] = function ($ch, $dltotal, $dlnow) use ($progressFile) {
                $state = [
                    'downloaded' => (int)$dlnow,
                    'total'      => (int)$dltotal,
                    'updated_at' => time(),
                ];
                @file_put_contents($progressFile, json_encode($state));
                return 0; // ادامه بده
            };
        } elseif (defined('CURLOPT_PROGRESSFUNCTION')) {
            $opts[CURLOPT_PROGRESSFUNCTION] = function ($resource, $dltotal, $dlnow) use ($progressFile) {
                $state = [
                    'downloaded' => (int)$dlnow,
                    'total'      => (int)$dltotal,
                    'updated_at' => time(),
                ];
                @file_put_contents($progressFile, json_encode($state));
                return 0;
            };
        }
    }

    if (file_exists(CA_BUNDLE)) {
        $opts[CURLOPT_SSL_VERIFYPEER] = true;
        $opts[CURLOPT_CAINFO]         = CA_BUNDLE;
    } else {
        $opts[CURLOPT_SSL_VERIFYPEER] = false;
    }

    curl_setopt_array($ch, $opts);
    $ok  = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    fclose($fp);

    if ($ok === false) {
        @unlink($tmp);
        throw new RuntimeException("Download failed: $err");
    }
    rename($tmp, $destPath);
}

/* ===========================
   ساخت thumbnail و placeholder
   =========================== */

function makeThumb(string $src, string $dest, int $maxW = 320, int $maxH = 320): void {
    $info = @getimagesize($src);
    if (!$info) return;
    [$w, $h, $type] = $info;
    $ratio = min($maxW / $w, $maxH / $h);
    $newW = max(1, (int)($w * $ratio));
    $newH = max(1, (int)($h * $ratio));
    switch ($type) {
        case IMAGETYPE_JPEG: $img = @imagecreatefromjpeg($src); break;
        case IMAGETYPE_PNG:  $img = @imagecreatefrompng($src);  break;
        case IMAGETYPE_GIF:  $img = @imagecreatefromgif($src);  break;
        default: return;
    }
    if (!$img) return;
    $thumb = imagecreatetruecolor($newW, $newH);
    imagecopyresampled($thumb, $img, 0,0,0,0, $newW,$newH, $w,$h);
    @imagejpeg($thumb, $dest, 82);
    imagedestroy($img);
    imagedestroy($thumb);
}

function makePlaceholderThumb(string $dest, string $label = 'FILE'): void {
    $dir = dirname($dest);
    if (!is_dir($dir)) mkdir($dir, 0775, true);
    $img = imagecreatetruecolor(320, 180);
    $bg  = imagecolorallocate($img, 30, 30, 30);
    imagefilledrectangle($img, 0,0,320,180,$bg);
    $txt = imagecolorallocate($img, 200,200,200);
    imagestring($img, 5, 10, 80, strtoupper($label), $txt);
    imagejpeg($img, $dest, 80);
    imagedestroy($img);
}

/* ===========================
   نمایش/دریافت لوگو یا فاوآیکون سایت
   =========================== */

/**
 * تلاش برای دانلود لوگو/فاوآیکون سایت و ذخیره در assets/logos/{host}.png
 * اولویت: apple-touch-icon → favicon.ico → آیکون‌های رایج در head
 * اگر پیدا نشود، یک تصویر متنی با نام سایت می‌سازد.
 * @return string مسیر نسبی فایل لوگو برای استفاده در UI
 */
function getSiteLogo(string $siteUrl): string {
    $host = parse_url($siteUrl, PHP_URL_HOST) ?: $siteUrl;
    $hostSlug = sanitizeSlug($host);
    $logoFile = LOGOS_PATH . '/' . $hostSlug . '.png';
    $logoRel  = 'assets/logos/' . $hostSlug . '.png';

    if (file_exists($logoFile)) {
        return $logoRel;
    }

    // تلاش برای استخراج از HTML
    try {
        $html = httpGet('https://' . $host);
    } catch (Throwable $e) {
        try {
            $html = httpGet('http://' . $host);
        } catch (Throwable $e2) {
            $html = '';
        }
    }

    $candidates = [];

    // apple-touch-icon
    if ($html && preg_match_all('~<link[^>]+rel=["\']apple-touch-icon[^"\']*["\'][^>]+href=["\']([^"\']+)["\']~i', $html, $m1)) {
        foreach ($m1[1] as $href) $candidates[] = absoluteUrl('https://' . $host, html_entity_decode($href));
    }
    // icon / shortcut icon
    if ($html && preg_match_all('~<link[^>]+rel=["\'](?:icon|shortcut icon)[^"\']*["\'][^>]+href=["\']([^"\']+)["\']~i', $html, $m2)) {
        foreach ($m2[1] as $href) $candidates[] = absoluteUrl('https://' . $host, html_entity_decode($href));
    }
    // fallback favicon.ico
    $candidates[] = 'https://' . $host . '/favicon.ico';

    foreach ($candidates as $iconUrl) {
        try {
            $data = httpGet($iconUrl, ["Accept: image/*"]);
            if ($data && strlen($data) > 100) {
                // تلاش برای تبدیل به PNG
                $img = @imagecreatefromstring($data);
                if ($img) {
                    @imagepng($img, $logoFile);
                    imagedestroy($img);
                    logMessage("Saved site logo for $host from $iconUrl");
                    return $logoRel;
                } else {
                    // اگر تصویر قابل خواندن نبود، همان داده را ذخیره کن
                    file_put_contents($logoFile, $data);
                    logMessage("Saved site icon (raw) for $host from $iconUrl");
                    return $logoRel;
                }
            }
        } catch (Throwable $e) {
            continue;
        }
    }

    // اگر چیزی پیدا نشد، تصویر نام سایت بساز
    makeSiteNameBadge($logoFile, $host);
    logMessage("Generated text badge for site $host");
    return $logoRel;
}

/**
 * ساخت یک تصویر ساده با نام سایت (Badge) برای جایگزین لوگو
 */
function makeSiteNameBadge(string $dest, string $siteName): void {
    $siteName = preg_replace('~^www\.~', '', strtolower($siteName));
    $text = sanitizeSlug($siteName);

    $img = imagecreatetruecolor(120, 120);
    $bg  = imagecolorallocate($img, 23, 26, 32);
    imagefilledrectangle($img, 0,0,120,120,$bg);

    $fg  = imagecolorallocate($img, 230, 234, 238);
    // استفاده از فونت داخلی GD (ساده، خوانا)
    imagestring($img, 5, 10, 50, $text, $fg);
    imagepng($img, $dest);
    imagedestroy($img);
}

/* ===========================
   مدیریت متادیتا
   =========================== */

function saveMeta(string $postId, array $meta, string $author = 'unknown', string $linkId = ''): void {
    $author = sanitizeSlug($author);
    $linkId = sanitizeSlug($linkId);
    $dir    = META_PATH . '/' . $author;
    if (!is_dir($dir)) mkdir($dir, 0775, true);

    $file = $dir . '/' . ($linkId ?: $postId) . '.json';
    file_put_contents($file, json_encode($meta, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
}

function loadAllMedia(): array {
    $items   = [];
    $authors = glob(META_PATH . '/*', GLOB_ONLYDIR);

    foreach ($authors as $authorDir) {
        $author = basename($authorDir);
        foreach (glob($authorDir . '/*.json') as $file) {
            $data = json_decode(file_get_contents($file), true);
            if (!$data || empty($data['items'])) continue;

            $linkId = basename($file, '.json');
            $count  = count($data['items']);
            $size   = 0;
            foreach ($data['items'] as $it) {
                if (!empty($it['file'])) {
                    $path = __DIR__ . '/' . $it['file'];
                    if (file_exists($path)) $size += filesize($path);
                }
            }

            foreach ($data['items'] as $it) {
                $items[] = [
                    'path'   => $it['file']   ?? '',
                    'thumb'  => $it['thumb']  ?? '',
                    'type'   => $it['type']   ?? 'image',
                    'name'   => $it['title']  ?? basename($file),
                    'url'    => $it['source'] ?? '',
                    'author' => $author,
                    'linkId' => $linkId,
                    'title'  => $it['title']  ?? '',
                    'ext'    => pathinfo($it['file'] ?? '', PATHINFO_EXTENSION),
                    'count'  => $count,
                    'size'   => $size,
                    'page'   => $data['source'] ?? ''
                ];
            }
        }
    }

    usort($items, fn($a,$b) => strcmp($b['name'], $a['name']));
    return $items;
}

function alreadyDownloaded(string $url): bool {
    $hash    = md5($url);
    $authors = glob(MEDIA_PATH . '/*', GLOB_ONLYDIR);

    foreach ($authors as $authorDir) {
        foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($authorDir)) as $fileinfo) {
            if ($fileinfo->isFile()) {
                if (strpos($fileinfo->getFilename(), $hash) !== false) {
                    return true;
                }
            }
        }
    }
    return false;
}

/* ===========================
   پوشه‌های نویسنده/لینک
   =========================== */

function ensureLinkDirs(string $author, string $linkId): array {
    $author = sanitizeSlug($author);
    $linkId = sanitizeSlug($linkId);
    $media  = MEDIA_PATH . '/' . $author . '/' . $linkId;
    $thumbs = THUMB_PATH . '/' . $author . '/' . $linkId;
    $meta   = META_PATH  . '/' . $author;

    foreach ([$media, $thumbs, $meta] as $dir) {
        if (!is_dir($dir)) mkdir($dir, 0775, true);
    }
    return [$media, $thumbs, $meta];
}

/* ===========================
   حذف امن فایل/پوشه‌ها با ثبت لاگ
   =========================== */

/**
 * حذف فایل مدیا یا تامنیل بر اساس مسیر نسبی (مثل storage/media/author/link/file.jpg)
 */
function deleteFileSafe(string $relativePath): bool {
    $full = __DIR__ . '/' . ltrim($relativePath, '/');
    if (!is_file($full)) {
        logMessage("DeleteFile: not found $relativePath");
        return false;
    }
    $ok = @unlink($full);
    logMessage("DeleteFile: " . ($ok ? "OK" : "FAIL") . " $relativePath");
    return $ok;
}

/**
 * حذف تمام محتوای یک لینک (پوشه‌ی media و thumbs، و متای مرتبط)
 */
function deleteLinkProfile(string $author, string $linkId): bool {
    $author = sanitizeSlug($author);
    $linkId = sanitizeSlug($linkId);

    $mediaDir  = MEDIA_PATH . '/' . $author . '/' . $linkId;
    $thumbDir  = THUMB_PATH . '/' . $author . '/' . $linkId;
    $metaFile  = META_PATH  . '/' . $author . '/' . $linkId . '.json';

    $okMedia = rrmdir($mediaDir);
    $okThumb = rrmdir($thumbDir);
    $okMeta  = is_file($metaFile) ? @unlink($metaFile) : true;

    logMessage("DeleteLinkProfile [$author/$linkId]: media=" . ($okMedia?'OK':'FAIL') . " thumbs=" . ($okThumb?'OK':'FAIL') . " meta=" . ($okMeta?'OK':'FAIL'));
    return $okMedia && $okThumb && $okMeta;
}

/**
 * حذف کامل یک پروفایل سایت (author): همه لینک‌ها، تامنیل‌ها و متا
 */
function deleteAuthorProfile(string $author): bool {
    $author = sanitizeSlug($author);
    $mediaAuthor = MEDIA_PATH . '/' . $author;
    $thumbAuthor = THUMB_PATH . '/' . $author;
    $metaAuthor  = META_PATH  . '/' . $author;

    $ok1 = rrmdir($mediaAuthor);
    $ok2 = rrmdir($thumbAuthor);
    $ok3 = rrmdir($metaAuthor);

    logMessage("DeleteAuthorProfile [$author]: media=" . ($ok1?'OK':'FAIL') . " thumbs=" . ($ok2?'OK':'FAIL') . " meta=" . ($ok3?'OK':'FAIL'));
    return $ok1 && $ok2 && $ok3;
}

/**
 * حذف بازگشتی پوشه
 */
function rrmdir(string $dir): bool {
    if (!is_dir($dir)) return true;
    $ok = true;
    foreach (scandir($dir) ?: [] as $item) {
        if ($item === '.' || $item === '..') continue;
        $path = $dir . '/' . $item;
        if (is_dir($path)) $ok = $ok && rrmdir($path);
        else $ok = $ok && @unlink($path);
    }
    return $ok && @rmdir($dir);
}

/* ===========================
   احراز هویت Reddit (برای پلاگین Reddit)
   =========================== */

function redditGetToken(string $client_id, string $client_secret, string $username, string $password): string {
    $ch = curl_init('https://www.reddit.com/api/v1/access_token');
    $opts = [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_USERPWD        => $client_id . ':' . $client_secret,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => [
            'grant_type' => 'password',
            'username'   => $username,
            'password'   => $password
        ],
        CURLOPT_HTTPHEADER     => ['User-Agent: MyDownloader/1.0 by reza'],
    ];
    if (file_exists(CA_BUNDLE)) {
        $opts[CURLOPT_SSL_VERIFYPEER] = true;
        $opts[CURLOPT_CAINFO]         = CA_BUNDLE;
    } else {
        $opts[CURLOPT_SSL_VERIFYPEER] = false;
    }

    curl_setopt_array($ch, $opts);
    $response = curl_exec($ch);
    if ($response === false) {
        $err = curl_error($ch);
        curl_close($ch);
        throw new RuntimeException('Failed to get token: ' . $err);
    }
    curl_close($ch);

    $data = json_decode($response, true);
    if (empty($data['access_token'])) throw new RuntimeException('No access_token in response: ' . $response);
    return $data['access_token'];
}