// assets/scripts.js
// نسخه‌ی واحد، مقاوم در برابر خطا، با مانیتورینگ امن JSON و هدایت پایدار

document.addEventListener("DOMContentLoaded", () => {
    /* ===========================
       Gauge با متن و انیمیشن نرم
       =========================== */
    function drawGauge(canvas, percent, label, text, color) {
        if (!canvas) return;
        const ctx = canvas.getContext("2d");
        if (!ctx) return;
        const radius = canvas.width / 2 - 10;
        ctx.clearRect(0, 0, canvas.width, canvas.height);

        // زمینه نیم‌دایره
        ctx.beginPath();
        ctx.arc(canvas.width/2, canvas.height/2, radius, Math.PI, 0);
        ctx.strokeStyle = "#444";
        ctx.lineWidth = 20;
        ctx.stroke();

        // مقدار
        const angle = Math.PI + (Math.max(0, Math.min(100, percent)) / 100) * Math.PI;
        ctx.beginPath();
        ctx.arc(canvas.width/2, canvas.height/2, radius, Math.PI, angle);
        ctx.strokeStyle = color;
        ctx.lineWidth = 20;
        ctx.stroke();

        // متن
        ctx.fillStyle = "#fff";
        ctx.font = "14px Vazirmatn, sans-serif";
        ctx.textAlign = "center";
        ctx.fillText(label, canvas.width/2, canvas.height/2 + 40);
        ctx.fillText(text ?? (percent + "%"), canvas.width/2, canvas.height/2 + 60);
    }

    function animateGauge(canvas, current, target, label, text, color) {
        let value = Number.isFinite(current) ? current : 0;
        const t = Number.isFinite(target) ? target : 0;
        const step = (t - value) / 20;
        function frame() {
            value += step;
            if ((step > 0 && value >= t) || (step < 0 && value <= t)) value = t;
            drawGauge(canvas, Math.round(value), label, text, color);
            if (value !== t) requestAnimationFrame(frame);
        }
        frame();
    }

    /* ===========================
       ابزار شبکه امن برای JSON
       =========================== */
    async function safeJson(response) {
        const contentType = (response.headers.get("Content-Type") || "").toLowerCase();
        const text = await response.text();
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${text.slice(0, 200)}`);
        }
        // اگر هدر JSON نیست، تلاش برای پارس وگرنه متن خطا بده
        if (!contentType.includes("application/json")) {
            try { return JSON.parse(text); }
            catch { throw new Error(`Non-JSON: ${text.slice(0, 200)}`); }
        }
        try { return JSON.parse(text); }
        catch { throw new Error(`Invalid JSON: ${text.slice(0, 200)}`); }
    }

    /* ===========================
       عناصر مشترک فرم و پیشرفت
       =========================== */
    const form = document.getElementById("downloadForm");
    const progressSection = document.getElementById("progressSection");
    const progressInfo = document.getElementById("progressInfo");
    const speedGauge = document.getElementById("speedGauge");
    const progressGauge = document.getElementById("progressGauge");

    let lastSpeed = 0;   // واحد: نمایش تا سقف 100
    let lastPercent = 0; // 0..100

    function formatSpeed(kbPerSec) {
        const mb = kbPerSec / 1024;
        if (mb < 1024) return mb.toFixed(2) + " MB/s";
        return (mb / 1024).toFixed(2) + " GB/s";
    }

    function startProgressPolling(progressId, redirectUrl) {
        if (!progressId) return;
        progressSection && progressSection.classList.remove("hidden");
        if (progressInfo) progressInfo.textContent = "دانلود آغاز شد...";

        const interval = setInterval(async () => {
            try {
                const r = await fetch("progress.php?id=" + encodeURIComponent(progressId));
                const p = await safeJson(r);

                // ساختار مورد انتظار: { total, downloaded, downloaded_str, total_str, speed }
                if (!p || !Number.isFinite(p.total) || p.total <= 0) return;

                const percent = Math.max(0, Math.min(100, Math.round((p.downloaded / p.total) * 100)));
                const speedKB = Number.isFinite(p.speed) ? p.speed : 0;
                const speedText = formatSpeed(speedKB);

                animateGauge(speedGauge, lastSpeed, Math.min(speedKB / 1024, 100), "سرعت", speedText, "#22c55e");
                animateGauge(progressGauge, lastPercent, percent, "پیشرفت", percent + "%", "#3b82f6");
                lastSpeed = Math.min(speedKB / 1024, 100);
                lastPercent = percent;

                if (progressInfo) {
                    const dStr = (p.downloaded_str ?? p.downloaded ?? "").toString();
                    const tStr = (p.total_str ?? p.total ?? "").toString();
                    progressInfo.textContent = `دانلود شده: ${dStr} / ${tStr} (${speedText})`;
                }

                if (percent >= 100) {
                    clearInterval(interval);
                    if (progressInfo) progressInfo.textContent = "✅ دانلود کامل شد!";
                    setTimeout(() => { window.location.href = redirectUrl; }, 2000);
                }
            } catch (err) {
                clearInterval(interval);
                if (progressInfo) progressInfo.textContent = "❌ خطا در مانیتورینگ: " + err.message;
            }
        }, 1000); // هر ثانیه
    }

    async function submitDownload(url, contentType = "") {
        const fd = new FormData();
        fd.set("url", url);
        if (contentType) fd.set("contentType", contentType);

        progressSection && progressSection.classList.remove("hidden");
        if (progressInfo) progressInfo.textContent = "در حال آماده‌سازی دانلود...";

        try {
            const r = await fetch("download.php", { method: "POST", body: fd });
            const data = await safeJson(r);

            if (data.error) {
                if (progressInfo) progressInfo.textContent = "❌ خطا: " + data.error;
                return;
            }
            if (data.requireContentType) {
                openContentTypeModal(url);
                return;
            }

            // آغاز مانیتورینگ
            startProgressPolling(data.progressId, data.redirect);
        } catch (err) {
            if (progressInfo) progressInfo.textContent = "❌ خطا: " + err.message;
        }
    }

    // اتصال فرم دانلود (در index.php). اگر در gallery فرم دیگری با id متفاوت دارید، آن را جداگانه وصل کنید.
    if (form) {
        form.addEventListener("submit", (e) => {
            e.preventDefault();
            const urlInput = form.querySelector('input[name="url"]') || form.querySelector('#url');
            const url = urlInput ? urlInput.value.trim() : "";
            if (!url) return;
            submitDownload(url);
        });
    }

    /* ===========================
       مودال انتخاب نوع محتوا
       =========================== */
    function openContentTypeModal(url) {
        const ctModal  = document.getElementById("contentTypeModal");
        const ctSelect = document.getElementById("contentTypeSelect");
        const ctOk     = document.getElementById("contentTypeOk");
        const ctCancel = document.getElementById("contentTypeCancel");

        if (!(ctModal && ctSelect && ctOk && ctCancel)) {
            if (progressInfo) progressInfo.textContent = "برای این لینک، نوع محتوا لازم است.";
            return;
        }

        ctModal.classList.remove("hidden");
        ctModal.setAttribute("aria-hidden", "false");
        ctModal.dataset.url = url;

        const onOk = () => {
            const type = ctSelect.value || "image";
            cleanup();
            submitDownload(url, type);
        };
        const onCancel = () => {
            cleanup();
            if (progressInfo) progressInfo.textContent = "عملیات لغو شد.";
        };
        function cleanup() {
            ctOk.removeEventListener("click", onOk);
            ctCancel.removeEventListener("click", onCancel);
            ctModal.classList.add("hidden");
            ctModal.setAttribute("aria-hidden", "true");
        }

        ctOk.addEventListener("click", onOk);
        ctCancel.addEventListener("click", onCancel);
    }

    /* ===========================
       پنجره تأیید حذف
       =========================== */
    function openConfirm(text, onOk) {
        const modal  = document.getElementById('confirmModal');
        const txt    = document.getElementById('confirmText');
        const ok     = document.getElementById('confirmOk');
        const cancel = document.getElementById('confirmCancel');

        if (!modal) return alert(text);

        txt && (txt.textContent = text);
        modal.classList.remove('hidden');
        modal.setAttribute('aria-hidden', 'false');

        const onOkWrap = () => {
            ok && ok.removeEventListener('click', onOkWrap);
            cancel && cancel.removeEventListener('click', onCancelWrap);
            modal.classList.add('hidden');
            modal.setAttribute('aria-hidden', 'true');
            onOk && onOk();
        };
        const onCancelWrap = () => {
            ok && ok.removeEventListener('click', onOkWrap);
            cancel && cancel.removeEventListener('click', onCancelWrap);
            modal.classList.add('hidden');
            modal.setAttribute('aria-hidden', 'true');
        };

        ok && ok.addEventListener('click', onOkWrap);
        cancel && cancel.addEventListener('click', onCancelWrap);
    }

    function deleteRequest(action, payload, onDone) {
        const params = new URLSearchParams();
        params.set('action', action);

        if (action === 'file') {
            params.set('target', payload.target);
        } else if (action === 'link') {
            params.set('author', payload.author);
            params.set('linkId', payload.linkId);
        } else if (action === 'author') {
            params.set('author', payload.author);
        }

        fetch('delete.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: params.toString()
        })
        .then(safeJson)
        .then(res => {
            if (res.error) {
                alert('❌ خطا: ' + res.error);
                return;
            }
            alert(res.message || (res.success ? '✅ حذف شد.' : '❌ حذف انجام نشد.'));
            onDone && onDone();
        })
        .catch(err => alert('❌ خطا در حذف: ' + err.message));
    }

    // اتصال دکمه‌های حذف (گالری)
    document.querySelectorAll('.btn-delete-file').forEach(btn => {
        btn.addEventListener('click', e => {
            e.preventDefault(); e.stopPropagation();
            const file = btn.dataset.file;
            if (!file) return;
            openConfirm('آیا مطمئن هستی که می‌خواهی این فایل را حذف کنی؟', () => {
                deleteRequest('file', { target: file }, () => location.reload());
            });
        });
    });

    document.querySelectorAll('.btn-delete-link').forEach(btn => {
        btn.addEventListener('click', e => {
            e.preventDefault(); e.stopPropagation();
            const author = btn.dataset.author;
            const linkId = btn.dataset.linkid;
            if (!author || !linkId) return;
            openConfirm('آیا مطمئن هستی که می‌خواهی کل محتوای این لینک را حذف کنی؟', () => {
                deleteRequest('link', { author, linkId }, () => {
                    window.location.href = 'gallery.php?author=' + encodeURIComponent(author);
                });
            });
        });
    });

    document.querySelectorAll('.btn-delete-author').forEach(btn => {
        btn.addEventListener('click', e => {
            e.preventDefault();
            const author = btn.dataset.author;
            if (!author) return;
            openConfirm('آیا مطمئن هستی که می‌خواهی کل محتوای این سایت را حذف کنی؟', () => {
                deleteRequest('author', { author }, () => {
                    window.location.href = 'gallery.php';
                });
            });
        });
    });

    /* ===========================
       ویوئر گالری
       =========================== */
    const viewer   = document.getElementById('viewer');
    const player   = document.getElementById('player');
    const closeBtn = document.getElementById('closeBtn');
    const prevBtn  = document.getElementById('prevBtn');
    const nextBtn  = document.getElementById('nextBtn');
    const deleteBtn= document.getElementById('deleteBtn');
    const cells    = Array.from(document.querySelectorAll('.cell[data-type]'));
    let currentIndex = -1;

    function showItem(index) {
        if (!cells.length) return;
        if (index < 0) index = cells.length - 1;
        if (index >= cells.length) index = 0;
        currentIndex = index;

        const cell = cells[index];
        const type = cell.dataset.type;
        const src  = cell.dataset.src;

        viewer && viewer.classList.remove('hidden');
        viewer && viewer.setAttribute('aria-hidden','false');
        if (player) player.innerHTML = '';

        if (type === 'image' || type === 'gif') {
            const img = document.createElement('img');
            img.src = src;
            img.className = 'fit';
            img.alt = 'media';
            player && player.appendChild(img);
            img.addEventListener('click', () => {
                if (img.classList.contains('zoomed')) {
                    img.classList.remove('zoomed'); img.classList.add('fit');
                } else {
                    img.classList.remove('fit'); img.classList.add('zoomed');
                }
            });
        } else if (type === 'video') {
            const vid = document.createElement('video');
            vid.src = src;
            vid.controls = true;
            vid.autoplay = true;
            vid.playsInline = true;
            vid.className = 'fit';
            player && player.appendChild(vid);
        } else {
            const a = document.createElement('a');
            a.href = src;
            a.textContent = 'دانلود فایل';
            a.target = '_blank';
            player && player.appendChild(a);
        }

        if (deleteBtn) deleteBtn.dataset.src = src;
    }

    cells.forEach((cell, i) => {
        cell.addEventListener('click', e => {
            const target = e.target;
            if (target && target.classList && target.classList.contains('btn-delete-file')) return;
            e.preventDefault();
            showItem(i);
        });
    });

    prevBtn && prevBtn.addEventListener('click', () => showItem(currentIndex - 1));
    nextBtn && nextBtn.addEventListener('click', () => showItem(currentIndex + 1));

    closeBtn && closeBtn.addEventListener('click', () => {
        viewer && viewer.classList.add('hidden');
        viewer && viewer.setAttribute('aria-hidden','true');
        if (player) player.innerHTML = '';
    });

    viewer && viewer.addEventListener('click', e => {
        if (e.target === viewer) {
            viewer.classList.add('hidden');
            viewer.setAttribute('aria-hidden','true');
            if (player) player.innerHTML = '';
        }
    });

    deleteBtn && deleteBtn.addEventListener('click', () => {
        const src = deleteBtn.dataset.src;
        if (!src) return;
        openConfirm('آیا مطمئن هستی که می‌خواهی این فایل را حذف کنی؟', () => {
            deleteRequest('file', { target: src }, () => location.reload());
        });
    });

    /* ===========================
       Gauge وضعیت دیسک (index.php)
       =========================== */
    const diskGauge = document.getElementById("diskGauge");
    if (diskGauge) {
        // اگر درصد استفاده از دیسک را در داده‌های سِروِر نداریم، فقط نمایشی خنثی
        drawGauge(diskGauge, 50, "فضای دیسک", "—", "#f59e0b");
    }
});